<?php
// /taxonomy.php

// --- 1. GESTIÓN DE SESIÓN Y SEGURIDAD ESTRICTA ---
session_start();
if (!isset($_SESSION['user_id'])) { header('Location: index.php'); exit; }
if ($_SESSION['user_rol'] !== 'superadmin') {
    http_response_code(403);
    die('Accés denegat. Aquesta secció és només per a administradors.');
}

require 'db.php';

// --- FUNCIÓN AUXILIAR PARA OBTENER SIEMPRE DATOS FRESCOS ---
function getTaxonomyData($db) {
    $catsStmt = $db->query("
        SELECT c.*, CASE WHEN EXISTS(SELECT 1 FROM categorias WHERE parent_id = c.id) THEN 1 ELSE 0 END AS has_children
        FROM categorias c ORDER BY c.parent_id, c.name
    ");
    $tagsStmt = $db->query("
        SELECT t.*, CASE WHEN EXISTS(SELECT 1 FROM tags WHERE parent_id = t.id) THEN 1 ELSE 0 END AS has_children
        FROM tags t ORDER BY t.parent_id, t.name
    ");
    return [
        'categories' => $catsStmt->fetchAll(),
        'tags' => $tagsStmt->fetchAll()
    ];
}

// --- 2. GESTOR DE PETICIONES AJAX (LÓGICA UNIFICADA Y ROBUSTA) ---
if (isset($_REQUEST['ajax'])) {
    header('Content-Type: application/json');
    $action = $_REQUEST['action'] ?? 'read_taxonomy';

    try {
        $response = ['status' => 'success'];

        switch($action) {
            case 'read_taxonomy':
                // Solo lee los datos
                break;
            case 'create_category':
            case 'create_tag':
                $table = ($action === 'create_category') ? 'categorias' : 'tags';
                $stmt = $db->prepare("INSERT INTO $table (name, parent_id) VALUES (?, ?)");
                $stmt->execute([$_POST['name'], empty($_POST['parent_id']) ? null : (int)$_POST['parent_id']]);
                $response['message'] = ucfirst(str_replace(['create_', '_'], ['', ' '], $action)) . ' creada.';
                break;
            case 'update_category':
            case 'update_tag':
                $table = ($action === 'update_category') ? 'categorias' : 'tags';
                if (isset($_POST['parent_id'])) {
                    $stmt = $db->prepare("UPDATE $table SET name = ?, parent_id = ? WHERE id = ?");
                    $stmt->execute([$_POST['name'], empty($_POST['parent_id']) ? null : (int)$_POST['parent_id'], (int)$_POST['id']]);
                } else {
                    $stmt = $db->prepare("UPDATE $table SET name = ? WHERE id = ?");
                    $stmt->execute([$_POST['name'], (int)$_POST['id']]);
                }
                $response['message'] = ucfirst(str_replace(['update_', '_'], ['', ' '], $action)) . ' actualitzada.';
                break;
            case 'delete_category':
            case 'delete_tag':
                $table = ($action === 'delete_category') ? 'categorias' : 'tags';
                $stmt = $db->prepare("DELETE FROM $table WHERE id = ?");
                $stmt->execute([(int)$_POST['id']]);
                $response['message'] = ucfirst(str_replace(['delete_', '_'], ['', ' '], $action)) . ' eliminada.';
                break;
        }

        // Después de cualquier acción (lectura o modificación), devuelve el estado MÁS RECIENTE de la BBDD
        $taxonomyData = getTaxonomyData($db);
        $response['categories'] = $taxonomyData['categories'];
        $response['tags'] = $taxonomyData['tags'];

        echo json_encode($response);

    } catch (Exception $e) {
        http_response_code(400);
        // Comprobar el código de error SQLSTATE para duplicados, que es '23000'
                if ($e instanceof PDOException && $e->getCode() == '23000') {
                    echo json_encode(['status' => 'error', 'message' => 'El nom ja existeix.']);
                } else {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }
    exit;
}

// --- 3. RENDERIZADO HTML ---
$page_title = "Gestionar Catàleg";
include 'partials/header.php';
?>
<main class="main-content container mt-4" style="max-width: 1420px;">
<h4><i class="bi bi-tags me-2"></i> Categories i Etiquetes</h4>
    <p class="text-muted">Des d'ací pots organitzar l'estructura de classificació que faran servir tots els fisioterapeutes.</p>
    <div class="row g-5">
        <div class="col-lg-6">
            <div class="card shadow-sm">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center mb-2">
                         <h5 class="mb-0"><i class="bi bi-diagram-3 me-2"></i>Categories Anatòmiques</h5>
                         <button class="btn btn-primary btn-sm" id="addCategoryBtn"><i class="bi bi-plus-circle"></i> Nova</button>
                    </div>
                    <div class="input-group input-group-sm">
                        <span class="input-group-text"><i class="bi bi-search"></i></span>
                        <input type="text" id="categorySearchInput" class="form-control" placeholder="Filtrar categories...">
                        <button class="clear-search-btn" type="button" id="clearCategorySearchBtn"><i class="bi bi-x"></i></button>
                    </div>
                </div>
                <div class="card-body" style="max-height: 70vh; overflow-y: auto;">
                    <ul id="category-list" class="list-group list-group-flush">
                        <li class="list-group-item text-center"><div class="spinner-border spinner-border-sm"></div></li>
                    </ul>
                </div>
            </div>
        </div>
        <div class="col-lg-6">
            <div class="card shadow-sm">
                 <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center mb-2">
                        <h5 class="mb-0"><i class="bi bi-tags me-2"></i>Etiquetes Funcionals</h5>
                        <button class="btn btn-primary btn-sm" id="addTagBtn"><i class="bi bi-plus-circle"></i> Nova</button>
                    </div>
                    <div class="input-group input-group-sm">
                         <span class="input-group-text"><i class="bi bi-search"></i></span>
                         <input type="text" id="tagSearchInput" class="form-control" placeholder="Filtrar etiquetes...">
                         <button class="clear-search-btn" type="button" id="clearTagSearchBtn"><i class="bi bi-x"></i></button>
                    </div>
                </div>
                <div class="card-body" style="max-height: 70vh; overflow-y: auto;">
                    <ul id="tag-list" class="list-group list-group-flush">
                         <li class="list-group-item text-center"><div class="spinner-border spinner-border-sm"></div></li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</main>

<div class="modal fade" id="categoryModal" tabindex="-1">
    <div class="modal-dialog"><div class="modal-content">
        <div class="modal-header"><h5 class="modal-title" id="categoryModalLabel">Nova Categoria</h5><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
        <div class="modal-body"><form id="categoryForm"><input type="hidden" name="id"><div class="mb-3"><label class="form-label">Nom</label><input type="text" class="form-control" name="name" required></div><div class="mb-3" id="category-parent-field"><label class="form-label">Categoria Pare</label><select class="form-select" name="parent_id"><option value="">Cap (Nivell Superior)</option></select></div><div class="modal-footer"><button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel·lar</button><button type="submit" class="btn btn-primary">Guardar</button></div></form></div>
    </div></div>
</div>

<div class="modal fade" id="tagModal" tabindex="-1">
    <div class="modal-dialog"><div class="modal-content">
        <div class="modal-header"><h5 class="modal-title" id="tagModalLabel">Nova Etiqueta</h5><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
        <div class="modal-body"><form id="tagForm"><input type="hidden" name="id"><div class="mb-3"><label class="form-label">Nom</label><input type="text" class="form-control" name="name" required></div><div class="mb-3" id="tag-parent-field"><label class="form-label">Etiqueta Pare</label><select class="form-select" name="parent_id"><option value="">Cap (Nivell Superior)</option></select></div><div class="modal-footer"><button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel·lar</button><button type="submit" class="btn btn-primary">Guardar</button></div></form></div>
    </div></div>
</div>

<?php include 'partials/footer.php'; ?>
<script>
$(document).ready(function() {
    const categoryModal = new bootstrap.Modal('#categoryModal');
    const tagModal = new bootstrap.Modal('#tagModal');
    let allCategories = [];
    let allTags = [];

    function updateAndRender(data) {
        allCategories = data.categories;
        allTags = data.tags;
        renderHierarchicalList('category', allCategories);
        renderHierarchicalList('tag', allTags);
    }

    function reloadData() {
        $.getJSON('taxonomy.php', { ajax: true, action: 'read_taxonomy' })
        .done(response => {
            if(response.status === 'success') {
                updateAndRender(response);
            }
        });
    }

    function renderHierarchicalList(type, items) {
        const searchTerm = $(`#${type}SearchInput`).val().toLowerCase();
        const container = $(`#${type}-list`);
        container.empty();

        // ================================================================== //
        // CORRECCIÓN APLICADA AQUÍ                                           //
        // ================================================================== //
        // Se elimina la lógica condicional. Ambas listas (categorías y etiquetas)
        // se renderizan ahora con la misma lógica jerárquica.
        let itemsToRender = items;
        if (searchTerm) {
            const idsToShow = new Set();
            const directMatches = items.filter(i => i.name.toLowerCase().includes(searchTerm));
            directMatches.forEach(match => {
                idsToShow.add(match.id);
                if (match.parent_id) {
                    let current = match;
                    while(current && current.parent_id) {
                        const parent = items.find(i => i.id == current.parent_id);
                        if(parent) { idsToShow.add(parent.id); current = parent; }
                        else { current = null; }
                    }
                } else {
                    items.forEach(potentialChild => {
                        if (potentialChild.parent_id == match.id) idsToShow.add(potentialChild.id);
                    });
                }
            });
            itemsToRender = items.filter(i => idsToShow.has(i.id));
        }

        const parents = itemsToRender.filter(i => !i.parent_id || !itemsToRender.some(p => p.id == i.parent_id));
        parents.sort((a, b) => a.name.localeCompare(b.name));

        if (itemsToRender.length === 0 && items.length > 0) {
            container.html(`<li class="list-group-item text-center text-muted">No s'han trobat resultats.</li>`);
        } else if (items.length === 0) {
            container.html(`<li class="list-group-item text-center text-muted">No hi ha ${type === 'category' ? 'categories' : 'etiquetes'} creades.</li>`);
        } else {
            parents.forEach(parent => renderNode(type, parent, itemsToRender, container, 0));
        }
    }

    function renderNode(type, node, allItems, container, level) {
        container.append(createTaxonomyItemHtml(type, node, level > 0));
        const children = allItems.filter(i => i.parent_id == node.id);
        children.sort((a, b) => a.name.localeCompare(b.name));
        children.forEach(child => renderNode(type, child, allItems, container, level + 1));
    }

    function createTaxonomyItemHtml(type, item, isChild) {
        const itemClass = isChild ? 'ps-4' : 'fw-bold';
        const namePrefix = isChild ? '↳ ' : '';
        return `<li class="list-group-item d-flex justify-content-between align-items-center ${itemClass}">
            <span>${namePrefix}${item.name}</span>
            <div class="btn-group">
                <button class="btn btn-sm btn-outline-secondary edit-${type}-btn" data-id="${item.id}" title="Editar"><i class="bi bi-pencil"></i></button>
                <button class="btn btn-sm btn-outline-danger delete-${type}-btn" data-id="${item.id}" title="Eliminar"><i class="bi bi-trash"></i></button>
            </div>
        </li>`;
    }

    function updateParentDropdown(type, items) {
        const form = $(`#${type}Form`);
        const select = form.find('[name="parent_id"]');
        const idBeingEdited = form.find('[name="id"]').val();

        select.html('<option value="">Cap (Nivell Superior)</option>');

        // Esta función, que controla el DESPLEGABLE, sigue filtrando
        // para mostrar solo los elementos de nivel superior. Esto es correcto.
        const topLevelItems = items.filter(i => !i.parent_id);
        topLevelItems.sort((a, b) => a.name.localeCompare(b.name));

        topLevelItems.forEach(item => {
            if (item.id != idBeingEdited) {
                select.append(new Option(item.name, item.id));
            }
        });
    }

    function setupSearch(inputId, clearBtnId, type) {
        const input = $(inputId);
        const clearBtn = $(clearBtnId);
        input.on('keyup', function() {
            clearBtn.toggle($(this).val().length > 0);
            renderHierarchicalList(type, type === 'category' ? allCategories : allTags);
        });
        clearBtn.on('click', function() { input.val('').trigger('keyup'); });
    }

    setupSearch('#categorySearchInput', '#clearCategorySearchBtn', 'category');
    setupSearch('#tagSearchInput', '#clearTagSearchBtn', 'tag');

    $('#addCategoryBtn, #addTagBtn').on('click', function() {
        const type = $(this).attr('id').includes('Category') ? 'category' : 'tag';
        const items = type === 'category' ? allCategories : allTags;
        const form = $(`#${type}Form`);
        const modal = $(`#${type}Modal`);
        form[0].reset();
        form.find('[name="id"]').val('');
        updateParentDropdown(type, items);
        $(`#${type}-parent-field`).show();
        modal.find('.modal-title').text(`Nova ${type === 'category' ? 'Categoria' : 'Etiqueta'}`);
        (type === 'category' ? categoryModal : tagModal).show();
    });

    $(document).on('click', '.edit-category-btn, .edit-tag-btn', function() {
        const type = $(this).hasClass('edit-category-btn') ? 'category' : 'tag';
        const items = type === 'category' ? allCategories : allTags;
        const id = $(this).data('id');
        const item = items.find(i => i.id == id);

        const form = $(`#${type}Form`);
        const modal = $(`#${type}Modal`);
        const parentField = $(`#${type}-parent-field`);

        form.find('[name="id"]').val(item.id);
        form.find('[name="name"]').val(item.name);

        updateParentDropdown(type, items);
        form.find('[name="parent_id"]').val(item.parent_id || '');

        if (!item.parent_id) {
            parentField.hide();
        } else {
            parentField.show();
        }

        modal.find('.modal-title').text(`Editar ${type === 'category' ? 'Categoria' : 'Etiqueta'}`);
        (type === 'category' ? categoryModal : tagModal).show();
    });

    function handleFormSubmit(e) {
        e.preventDefault();
        const form = $(this);
        const type = form.attr('id').includes('category') ? 'category' : 'tag';
        const action = form.find('[name="id"]').val() ? `update_${type}` : `create_${type}`;

        $.post('taxonomy.php', `ajax=true&action=${action}&${form.serialize()}`, (res) => {
            if (res.status === 'success') {
                showToast(res.message, 'success');
                (type === 'category' ? categoryModal : tagModal).hide();
                updateAndRender(res);
            }
        }).fail(xhr => showToast(xhr.responseJSON.message, 'danger'));
    }

    $('#categoryForm, #tagForm').on('submit', handleFormSubmit);

    function handleDeleteClick() {
        const type = $(this).hasClass('delete-category-btn') ? 'category' : 'tag';
        const id = $(this).data('id');
        if (confirm(`Estàs segur/a que vols eliminar aquesta ${type}? El canvi serà permanent i no es podrà desfer.`)) {
            $.post('taxonomy.php', { ajax: true, action: `delete_${type}`, id }, (res) => {
                if(res.status === 'success') {
                    showToast(res.message, 'success');
                    updateAndRender(res);
                }
            })
            .fail(xhr => showToast(xhr.responseJSON.message, 'danger'));
        }
    }

    $(document).on('click', '.delete-category-btn, .delete-tag-btn', handleDeleteClick);

    reloadData();
});
</script>
